#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdint.h>
#include <input/mtouch_driver.h>
#include <input/parseopts.h>
#include <sys/select.h>
#include <pthread.h>
#include <input/inputtrace.h>
#include <input/mtouch_log.h>
#include <string.h>
#include <errno.h>
#include <sys/slog2.h>

#include "himax_platform.h"
#include "himax_touch_driver.h"

extern void himax_mcu_system_reset(himax_dev_t *himax);

#if 0
int hx83192_sense_off(himax_dev_t *himax)
{
        uint8_t cnt = 0;
        uint8_t tmp_addr[4] = {0};
        uint8_t tmp_data[4] = {0};
        uint8_t cMax = 7;
        uint8_t check = 0x87;
        uint8_t w_data[1];

        usleep(280*1000);
        tmp_addr[0] = 0xA8;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x90;
        himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

        if (tmp_data[0] != 0x0C) {
                tmp_addr[0] = 0x5C;
                tmp_addr[1] = 0x00;
                tmp_addr[2] = 0x00;
                tmp_addr[3] = 0x90;
                cnt = 0;
                do {
                        tmp_data[0] = 0xA5;
                        tmp_data[1] = 0x00;
                        tmp_data[2] = 0x00;
                        tmp_data[3] = 0x00;
                        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);
                        usleep(20*1000);
                        himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

                        mtouch_info(HIMAX_DEVNAME, "Check 9000005C data[0]=%X\n", tmp_data[0]);
                        if (cnt++ >= cMax) {
                                if (tmp_data[0] == 0xA5) {
                                    /* Firmware is likely invalid, continue on to the next step in setting safe mode */
                                    break;
                                }

                                return -1;
                        }
                } while (tmp_data[0] != check);
        }

        cnt = 0;

        do {
                mtouch_info(HIMAX_DEVNAME, "cnt is %d\n", cnt);
                w_data[0] = 0x27;
                if (himax_i2c_write(himax->i2c_fd, 0x31, w_data, 1) != EOK) {
                        mtouch_error(HIMAX_DEVNAME, "i2c write fail!\n");
                        return EIO;
                }
                w_data[0] = 0x95;
                if (himax_i2c_write(himax->i2c_fd, 0x32, w_data, 1) != EOK) {
                        mtouch_error(HIMAX_DEVNAME, "i2c write fail!\n");
                        return EIO;
                }

                tmp_data[0] = 0x00;
                w_data[0] = 0x00;
                if (himax_i2c_write(himax->i2c_fd, 0x31, w_data, 1) != EOK) {
                        mtouch_error(HIMAX_DEVNAME, "i2c write fail!\n");
                        return EIO;
                }
                usleep(100);
                w_data[0] = 0x27;
                if (himax_i2c_write(himax->i2c_fd, 0x31, w_data, 1) != EOK) {
                        mtouch_error(HIMAX_DEVNAME, "i2c write fail!\n");
                        return EIO;
                }
                w_data[0] = 0x95;
                if (himax_i2c_write(himax->i2c_fd, 0x32, w_data, 1) != EOK) {
                        mtouch_error(HIMAX_DEVNAME, "i2c write fail!\n");
                        return EIO;
                }

                tmp_addr[0] = 0xA8;
                tmp_addr[1] = 0x00;
                tmp_addr[2] = 0x00;
                tmp_addr[3] = 0x90;
                himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);
                mtouch_info(HIMAX_DEVNAME, "Check enter_save_mode data[0]=%X\n", tmp_data[0]);

                if (tmp_data[0] == 0x0C) {
                    return EOK;
                } else if (cnt == 6) {
                        usleep(10000);
                        himax_mcu_system_reset(himax);
                }
        } while (cnt++ < 15);

        return EOK;
}
#endif

/* Note: We have only a rough idea what these commands are.  Himax won't give us documentation on the registers so we
 * have to use their sample code from the docs blindly. The flow chart on Section 5.2.1 (Page 47) of the
 * HX83192A_A_Application_Note_V1.4.pdf lists roughly what the commands are doing.  Those are documented along with
 * the commands below. */
int
himax_chip_erase(himax_dev_t *himax)
{
        uint8_t tmp_addr[4] = {0};
        uint8_t tmp_data[4] = {0};

    /* Initial Power Saving Level */
    tmp_addr[0] = 0xA0;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x90;

    tmp_data[0] = 0;
    tmp_data[1] = 0;
    tmp_data[2] = 0;
    tmp_data[3] = 0;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* SPI Format */
    tmp_addr[0] = 0x10;
    tmp_addr[1] = 0;
    tmp_addr[2] = 0;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x80;
    tmp_data[1] = 0x07;
    tmp_data[2] = 0x02;
    tmp_data[3] = 0x00;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* Flash Write Enable */
    tmp_addr[0] = 0x20;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x00;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x47;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    tmp_addr[0] = 0x24;
    tmp_addr[1] = 0;
    tmp_addr[2] = 0;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x06;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x00;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* Chip Erase*/
    tmp_addr[0] = 0x24;
    tmp_addr[1] = 0;
    tmp_addr[2] = 0;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0xC7;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x00;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    delay (800);

    return 0;
}

/* Note: We have only a rough idea what these commands are.  Himax won't give us documentation on the registers so we
 * have to use their sample code from the docs blindly. The flow chart on Section 5.2.1 (Page 47) of the
 * HX83192A_A_Application_Note_V1.4.pdf lists roughly what the commands are doing.  Those are documented along with
 * the commands below. */
/* WIP = Work in Progress */
int
himax_wait_wip(himax_dev_t *himax, useconds_t usec)
{
    int retry_limit = 0;

        uint8_t tmp_addr[4] = {0};
        uint8_t tmp_data[4] = {0};

    /* SPI Format */
    tmp_addr[0] = 0x10;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x80;
    tmp_data[1] = 0x07;
    tmp_data[2] = 0x02;
    tmp_data[3] = 0x00;

    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    while (1) {
        if (retry_limit > 100) {
            mtouch_error(himax->log_name, "WIP retry exeeded limit");
            error_memory("Himax_Touch: WIP retry exeeded limit");
            return -1;
        }

        /* SPI Transfer Control */
        tmp_addr[0] = 0x20;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x03;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x42;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* SPI Command */
        tmp_addr[0] = 0x24;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x05;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x00;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* Read WIP */
        tmp_addr[0] = 0x2C;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;
        himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

        if (!(tmp_data[0] & 0x01))
            break;

        retry_limit++;

        //safe_usleep(usec);
        delay (1);
    }

    return 0;
}

int
himax_open_firmware_file(himax_dev_t *himax, char *firmware_file, uint32_t **firmware)
{
    FILE *fp = NULL;
    int file_size = 0;
    int bytes_read = 0;

    /* Open file */
    if (NULL == firmware_file) {
            mtouch_error (himax->log_name, "No firmware file specified, cannot update firmware");
            error_memory("Himax_Touch: No firmware file specified, cannot update firmware");
            return -1;
    }

    mtouch_info (himax->log_name, "Opening %s", firmware_file);

    fp = fopen (firmware_file, "r");
    if (NULL == fp) {
            mtouch_error (himax->log_name, "Error opening firmware file %s: %d", firmware_file, errno);
            error_memory("Himax_Touch: Error opening firmware file %s: %d", firmware_file, errno);
            return -1;
    }

    /* Figure out how big the file is */
    fseek (fp, 0L, SEEK_END);
    file_size = ftell (fp);
    if (file_size < 0){
        mtouch_error (himax->log_name, "Invalid file size %d", file_size);
        error_memory("Himax_Touch: Invalid file size %d", file_size);
        fclose(fp);
        return -1;
    }

    rewind (fp);

        mtouch_info(himax->log_name, "Firmware file is %d bytes in size", file_size);

    if (*firmware != NULL) {
        free (*firmware);
        *firmware = NULL;
    }

    *firmware = calloc (file_size, sizeof (uint16_t));

    if (*firmware == NULL) {
        mtouch_error (himax->log_name, "Failed to allocate memory for firmware file.");
        error_memory("Himax_Touch: Failed to allocate memory for firmware file.");
        fclose(fp);
        return -1;
    }

    /* Read in the entire firmware file */
    bytes_read = fread(*firmware, sizeof(uint16_t), file_size, fp);

    if ((bytes_read * sizeof(uint16_t)) != file_size) {
        mtouch_error (himax->log_name, "%s was unable to read in firmware file correctly.  Firmware file size is %d and bytes read in %d", __FUNCTION__, file_size, bytes_read);
        error_memory("Himax_Touch: %s was unable to read in firmware file correctly.  Firmware file size is %d and bytes read in %d", __FUNCTION__, file_size, bytes_read);
        free (*firmware);
        fclose(fp);
        return -1;
    }

    fclose(fp);
    return file_size;
}

int
himax_program_firmware(himax_dev_t *himax, uint32_t *firmware, int remaining_bytes)
{
    uint32_t flash_address = 0x00000000;

        uint8_t tmp_addr[4] = {0};
        uint8_t tmp_data[4] = {0};

    int i = 0;

    /* Set SPI Speed */
    tmp_addr[0] = 0x40;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x03;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x00;
    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* SPI TX FIFO Reset */
    tmp_addr[0] = 0x30;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x04;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x00;
    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* Polling SPI Status */
    do {
                tmp_addr[0] = 0x34;
                tmp_addr[1] = 0x00;
                tmp_addr[2] = 0x00;
                tmp_addr[3] = 0x80;
                himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

        mtouch_info(himax->log_name, "%d Check 0x80000034 data[0]=%X\n", __LINE__, tmp_data[0]);
    } while ((tmp_data[0] & 0x04) != 0);

    /* Set SPI Format */
    tmp_addr[0] = 0x10;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x00;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x80;
    tmp_data[1] = 0x07;
    tmp_data[2] = 0x02;
    tmp_data[3] = 0x00;
    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    while (remaining_bytes) {
        /* Flash Write Enable */
        tmp_addr[0] = 0x20;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x00;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x47;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        tmp_addr[0] = 0x24;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x06;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x00;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* Polling SPI Status */
        do {
            tmp_addr[0] = 0x34;
            tmp_addr[1] = 0x00;
            tmp_addr[2] = 0x00;
            tmp_addr[3] = 0x80;
            himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

            mtouch_info(himax->log_name, "%d Check 0x80000034 data[0]=%X\n", __LINE__, tmp_data[0]);
        } while ((tmp_data[0] & 0x01) != 0);

        /* WEL Write Control */
        tmp_addr[0] = 0x20;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x00;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x42;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        tmp_addr[0] = 0x24;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x05;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x00;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* Polling SPI Status */
        do {
            tmp_addr[0] = 0x34;
            tmp_addr[1] = 0x00;
            tmp_addr[2] = 0x00;
            tmp_addr[3] = 0x80;
            himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);
            mtouch_info(himax->log_name, "%d Check 0x80000034 data[0]=%X\n", __LINE__, tmp_data[0]);
        } while ((tmp_data[0] & 0x01) != 0);

        /* */
        tmp_addr[0] = 0x2C;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;
        himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

        if (0 == (tmp_data[0] & 0x02)) {
            mtouch_critical(himax->log_name, "WEL Fail");
            return -1;
        }

        /* Set 256 Bytes Page Write */
        tmp_addr[0] = 0x20;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x00;
        tmp_data[1] = 0xF0;
        tmp_data[2] = 0x0F;
        tmp_data[3] = 0x61;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* Set SPI Address */
        tmp_addr[0] = 0x28;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = flash_address & 0xFF;
        tmp_data[1] = (flash_address >> 8) & 0xFF;
        tmp_data[2] = (flash_address >> 16) & 0xFF;
        tmp_data[3] = (flash_address >> 24) & 0xFF;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        /* Write Control */
        tmp_addr[0] = 0x24;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x80;

        tmp_data[0] = 0x02;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x00;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

        mtouch_info(himax->log_name, "Writing to flash address 0x%08x", flash_address);

        for (i = 0; i < (256 / 16); i++) {
            mtouch_info(himax->log_name, "Writing chunk %d of %d", i, (256/16));

            /* */
            tmp_addr[0] = 0x2C;
            tmp_addr[1] = 0x00;
            tmp_addr[2] = 0x00;
            tmp_addr[3] = 0x80;

            tmp_data[0] = firmware[i] & 0xFF;
            tmp_data[1] = (firmware[i] >> 8) & 0xFF;
            tmp_data[2] = (firmware[i] >> 16) & 0xFF;
            tmp_data[3] = (firmware[i] >> 24) & 0xFF;
            himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

            /* Polling SPI Status */
            do {
                tmp_addr[0] = 0x34;
                tmp_addr[1] = 0x00;
                tmp_addr[2] = 0x00;
                tmp_addr[3] = 0x80;
                himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);
                mtouch_info(himax->log_name, "%d Check 0x80000034 data[2]=%X\n", __LINE__, tmp_data[2]);
            } while ((tmp_data[2] & 0x40) == 0);
        }

        if (-1 == himax_wait_wip(himax, 1000)) /* Poll every 1ms */
            return -1;

        /* Add 256 to flash address */
        flash_address += 256;

        remaining_bytes -= 256;
    }

    return 0;
}

int
himax_check_flash(himax_dev_t *himax)
{
    //int retry_limit = 0;
    //uint32_t start_address = 0x00000000;

    uint8_t tmp_addr[4] = {0};
    uint8_t tmp_data[4] = {0};

    /* Set Start Address */
    tmp_addr[0] = 0x20;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x05;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x00;
    tmp_data[1] = 0x00;
    tmp_data[2] = 0x00;
    tmp_data[3] = 0x00;
    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* Set Length */
    tmp_addr[0] = 0x28;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x05;
    tmp_addr[3] = 0x80;

    tmp_data[0] = 0x00;
    tmp_data[1] = 0x80;
    tmp_data[2] = 0x99;
    tmp_data[3] = 0x00;
    himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    /* wait until CRC results a ready */
    do {
        tmp_addr[0] = 0x00;
        tmp_addr[1] = 0x00;
        tmp_addr[2] = 0x05;
        tmp_addr[3] = 0x80;
        himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);
        mtouch_info(himax->log_name, "%d Check 0x80050000 data[0]=%X\n", __LINE__, tmp_data[0]);
    } while ((tmp_data[0] & 0x01) == 0x01);

    tmp_addr[0] = 0x18;
    tmp_addr[1] = 0x00;
    tmp_addr[2] = 0x05;
    tmp_addr[3] = 0x80;
    himax_register_read(himax->i2c_fd, tmp_addr, 4, tmp_data);

    /* Modifying to solve coverity issue, needs recheck */
    //if (((tmp_data[3] << 24) & (tmp_data[2] << 16) & (tmp_data[1] << 8) & tmp_data[0]) != 0x00000000) {
	if (((tmp_data[3]) & (tmp_data[2]) & (tmp_data[1]) & tmp_data[0]) != 0x00000000) {
        return -1;
    }

    return 0;
}

int
enable_flash_reload(himax_dev_t *himax)
{
    uint8_t tmp_addr[4] = {0};
        uint8_t tmp_data[4] = {0};

        /* Set SPI Speed */
        tmp_addr[0] = 0x00;
        tmp_addr[1] = 0x7F;
        tmp_addr[2] = 0x00;
        tmp_addr[3] = 0x10;

        tmp_data[0] = 0x00;
        tmp_data[1] = 0x00;
        tmp_data[2] = 0x00;
        tmp_data[3] = 0x00;
        himax_register_write(himax->i2c_fd, tmp_addr, tmp_data);

    return 0;
}


int
update_firmware(himax_dev_t *himax, char *firmware_file)
{
   /* changing the data type of frimware from uint16 to uint32 to resolev coveryty issue,
      needs to be rechecked during FW update implementation */
    uint32_t *firmware = NULL;
    int file_size = 0;
    int ret = -1;

    /* Check to see if the firmware file is accessible and if so open it */
    if (-1 != (file_size = himax_open_firmware_file(himax, firmware_file, &firmware))) {
        mtouch_info(himax->log_name, "Firmware file opened successfully, attempting to enter safe mode...");

        if (EOK == hx83192_sense_off(himax)) {
            mtouch_info(himax->log_name, "Hardware is now in safe mode, attempting to erase chip...");

            /*himax_disable_protected_mode
            if (EOK == himax_disable_protected_mode(himax)) {
                            mtouch_info(HIMAX_DEVNAME, "Disable protected mode... ");*/

            if (EOK == himax_chip_erase(himax)) {
                mtouch_info(himax->log_name, "Waiting for chip erase operation to complete...");

                if (EOK == himax_wait_wip(himax, 100000)) {
                    mtouch_info(himax->log_name, "Chip erase complete, attempting to program firmware...");

                    if (EOK == himax_program_firmware(himax, firmware, file_size)) {
                        mtouch_info(himax->log_name, "Firmware writing complete, checking flash...");

                        if (EOK == himax_check_flash(himax)) {
                            mtouch_info(himax->log_name, "Flash checking complete, reloading flash...");

                            /* Enable Flash Reload */
                            if (EOK == enable_flash_reload(himax)) {
                                mtouch_info(himax->log_name, "Resetting controller...");

                                /* Perform soft reset */
                                            himax_mcu_system_reset(himax);
                            }
                        }
                    }
                }
            }
        }
    }


    if (NULL != firmware)
        free (firmware);

    return ret;
}
